<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\DepositWithdrawal;
use App\UserWallet;
use Illuminate\Support\Facades\DB;

class DepositWithdrawalController extends Controller
{
    /**
     * Pending deposits/withdrawals
     */
    public function index()
    {
        $pending = DepositWithdrawal::with('user')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.deposits.index', compact('pending'));
    }

    /**
     * All deposits/withdrawals
     */
    public function all()
    {
        $transactions = DepositWithdrawal::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.deposits.all', compact('transactions'));
    }

    /**
     * Approve withdrawal
     */
    public function approveWithdrawal(Request $request, $id)
    {
        $withdrawal = DepositWithdrawal::findOrFail($id);

        if ($withdrawal->type !== 'withdrawal') {
            return back()->with('error', 'این تراکنش برداشت نیست');
        }

        if ($withdrawal->status !== 'pending') {
            return back()->with('error', 'این تراکنش قبلاً پردازش شده');
        }

        try {
            DB::beginTransaction();

            // Update status
            $withdrawal->status = 'completed';
            $withdrawal->completed_at = now();
            $withdrawal->admin_note = $request->note;
            $withdrawal->txid = $request->txid;
            $withdrawal->save();

            // Remove from locked balance
            $wallet = UserWallet::where('user_id', $withdrawal->user_id)
                ->where('currency', $withdrawal->currency)
                ->first();

            if ($wallet) {
                $wallet->completeTrade($withdrawal->amount);
            }

            DB::commit();

            return back()->with('success', 'برداشت تأیید شد');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'خطا: ' . $e->getMessage());
        }
    }

    /**
     * Reject withdrawal
     */
    public function rejectWithdrawal(Request $request, $id)
    {
        $withdrawal = DepositWithdrawal::findOrFail($id);

        if ($withdrawal->type !== 'withdrawal') {
            return back()->with('error', 'این تراکنش برداشت نیست');
        }

        try {
            DB::beginTransaction();

            // Update status
            $withdrawal->status = 'cancelled';
            $withdrawal->admin_note = $request->note;
            $withdrawal->save();

            // Unlock balance
            $wallet = UserWallet::where('user_id', $withdrawal->user_id)
                ->where('currency', $withdrawal->currency)
                ->first();

            if ($wallet) {
                $wallet->unlockAmount($withdrawal->amount);
            }

            DB::commit();

            return back()->with('success', 'برداشت رد شد و موجودی بازگردانده شد');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'خطا: ' . $e->getMessage());
        }
    }

    /**
     * Confirm deposit
     */
    public function confirmDeposit(Request $request, $id)
    {
        $deposit = DepositWithdrawal::findOrFail($id);

        if ($deposit->type !== 'deposit') {
            return back()->with('error', 'این تراکنش واریز نیست');
        }

        try {
            DB::beginTransaction();

            // Update status
            $deposit->status = 'completed';
            $deposit->completed_at = now();
            $deposit->admin_note = $request->note;
            $deposit->save();

            // Add to balance
            $wallet = UserWallet::firstOrCreate([
                'user_id' => $deposit->user_id,
                'currency' => $deposit->currency
            ]);

            $wallet->addBalance($deposit->amount);

            DB::commit();

            return back()->with('success', 'واریز تأیید شد');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'خطا: ' . $e->getMessage());
        }
    }
}
