<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class DiagnosticController extends Controller
{
    /**
     * Test price fetching and display diagnostic information
     */
    public function testPrices()
    {
        $diagnostics = [];
        
        try {
            // Test 1: Check if CoinPriceService exists
            $diagnostics['service_exists'] = class_exists('\App\Services\CoinPriceService');
            
            // Test 2: Create service instance
            $priceService = new \App\Services\CoinPriceService();
            $diagnostics['service_created'] = true;
            
            // Test 3: Get settings
            $diagnostics['settings'] = [
                'usdt_price_source' => getSetting('usdt_price_source') ? getSetting('usdt_price_source')->value : 'not set',
                'tetherland_api_url' => getSetting('tetherland_api_url') ? getSetting('tetherland_api_url')->value : 'not set',
                'tetherland_private_key' => getSetting('tetherland_private_key') ? 'SET (hidden)' : 'NOT SET',
                'USDT_price_buy' => getSetting('USDT_price_buy') ?? 'not set',
                'USDT_price_sell' => getSetting('USDT_price_sell') ?? 'not set',
            ];
            
            // Test 4: Test Tetherland API directly
            $diagnostics['tetherland_test'] = $this->testTetherlandAPI();
            
            // Test 5: Fetch all prices
            $allPrices = $priceService->getAllPrices();
            $diagnostics['prices_fetched'] = count($allPrices);
            $diagnostics['sample_prices'] = array_slice($allPrices, 0, 3, true);
            
            // Test 6: Get cache status
            $diagnostics['cache_status'] = $priceService->getCacheStatus();
            
            // Test 7: Check CoinEx API
            $diagnostics['coinex_test'] = $this->testCoinExAPI();
            
            return response()->json([
                'status' => 'success',
                'diagnostics' => $diagnostics
            ], 200, [], JSON_PRETTY_PRINT);
            
        } catch (\Exception $e) {
            $diagnostics['error'] = $e->getMessage();
            $diagnostics['trace'] = $e->getTraceAsString();
            
            return response()->json([
                'status' => 'error',
                'diagnostics' => $diagnostics
            ], 500, [], JSON_PRETTY_PRINT);
        }
    }
    
    /**
     * Test Tetherland API connection
     */
    private function testTetherlandAPI()
    {
        try {
            $apiUrl = getSetting('tetherland_api_url');
            $privateKey = getSetting('tetherland_private_key');
            
            $endpoint = $apiUrl ? $apiUrl->value : 'https://service.tetherland.com/api/v5/user/converter/exchange';
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $endpoint);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            
            $headers = ['Accept: application/json'];
            if ($privateKey && !empty($privateKey->value)) {
                $key = trim($privateKey->value);
                $headers[] = 'Authorization: Bearer ' . $key;
            }
            
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            $data = json_decode($response, true);
            
            return [
                'endpoint' => $endpoint,
                'http_code' => $httpCode,
                'curl_error' => $curlError ?: 'none',
                'response' => $data,
                'success' => ($httpCode === 200 && isset($data['status']) && $data['status'] === true)
            ];
            
        } catch (\Exception $e) {
            return [
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Test CoinEx API connection
     */
    private function testCoinExAPI()
    {
        try {
            $response = file_get_contents('https://api.coinex.com/v1/market/ticker/all');
            $data = json_decode($response, true);
            
            return [
                'success' => isset($data['code']) && $data['code'] === 0,
                'pairs_count' => isset($data['data']['ticker']) ? count($data['data']['ticker']) : 0,
                'sample_pair' => isset($data['data']['ticker']['BTCUSDT']) ? $data['data']['ticker']['BTCUSDT'] : 'not found'
            ];
            
        } catch (\Exception $e) {
            return [
                'error' => $e->getMessage()
            ];
        }
    }
}
