<?php

namespace App\Http\Controllers\Admin;

use App\Category;
use App\Product;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use phpDocumentor\Reflection\Location;


class ProductController extends Controller
{
    public function index()
    {
        $products = Product::query()
            ->with(['category' => function ($model) {
                $model->where('active', 1)
                    ->select('id', 'title');
            }])
            ->select([
                'products.id',
                'products.title',
                'products.image',
                'products.product_price',
                'category_id',
            ])
            ->withSum(['productOrder as sum_final_price' => function($model) {
                $model->where('status', 'done');
            }], 'final_price')
            ->withSum(['productOrder as sum_delivery_price' => function($model) {
                $model->where('status', 'done');
            }], 'delivery_price')
            ->withSum(['productOrder as sum_quantity' => function($model) {
                $model->where('status', 'done');
            }], 'quantity')
            ->where('products.approved', 1)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('admin.shop.products', [ 'products' => $products ]);
    }

    public function ProductIndex(Request $request)
    {
        $productId = (int) $request->id;

        if(isset($productId))
            $product = Product::query()->find($productId);

        $categories = Category::query()
            ->select('id','title')
            ->where('active', 1)
            ->get();

        return view('admin.shop.add-product', [
            'categories'    => $categories,
            'product'       => $product
        ]);
    }

    public function addProduct(Request $request)
    {
        if(isset($request->product_id)) {
            $product    = Product::query()->find($request->product_id);
            $msg        = 'محصول با موفقیت ویرایش شد';
        }

        $validator = \Validator::make($request->all(), [
            'title' => [
                'required',
                'max:255',
                Rule::unique('products')->ignore($request->product_id),
            ],
            'category_id'   => 'required',
            'image'         => Rule::requiredIf(isset($product) ? $product->image == null : true),
            'quantity'      => 'required|numeric|digits_between:1,4',
            'buy_price'     => 'required|numeric|digits_between:1,9',
            'price'         => 'required|numeric|digits_between:1,9',
            'description'   => 'nullable',
        ], [
            'title.required'    => 'نام محصول را وارد نمایید',
            'title.max'         => 'نام محصول باید حداکثر :max کاراکتر باشد',
            'title.unique'      => 'این نام قبلا ثبت شده است',
            'category_id.required'  => 'دسته بندی محصول را وارد نمایید',
            'quantity.required'     => 'تعداد محصول را وارد نمایید',
            'image.required'        => 'تصویر محصول را وارد کنید',
            'buy_price.required'    => 'قیمت خرید محصول را وارد نمایید',
            'price.required'    => 'قیمت فروش محصول را وارد نمایید',
            'quantity.numeric'  => 'تعداد محصول را به عدد وارد نمایید',
            'buy_price.numeric' => 'قیمت خرید محصول را به عدد وارد نمایید',
            'price.numeric'     => 'قیمت فروش محصول را به عدد وارد نمایید',
            'quantity.digits_between'   => 'تعداد ارقام برای تعداد محصول معتبر نمیباشند',
            'price.digits_between'      => 'تعداد ارقام برای قیمت فروش محصول معتبر نمیباشند',
            'buy_price.digits_between'  => 'تعداد ارقام برای قیمت خرید محصول معتبر نمیباشند',
        ]);

        if ($validator->fails())
            return redirect()->back()->withErrors($validator->errors())->withInput();

        if(!isset($request->product_id)) {
            $product    = new Product();
            $msg    = 'محصول جدید با موفقیت ایجاد شد';
        }

        if($request->hasFile('image')) {
            if(isset($request->product_id) && File::exists($product->image))
                File::delete($product->image);

            $file = $request->file('image');
            $extension = $file->getClientOriginalExtension();
            $just_name = str_replace(" ", "-", pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
            $just_name_hash = str_replace("/", "", $just_name . time());
            $name = $just_name_hash . '.' . $extension;

            $request->file('image')->move(LARAVEL_PUBLIC_DIR  . '/images/products/', $name);
        }

        $product->title = $request->title;
        $product->description   = $request->description;
        $product->category_id   = $request->category_id;
        $product->image     = isset($name) ? 'images/products/' . $name : $product->image;
        $product->quantity  = $request->quantity;
        $product->buy_price = $request->buy_price;
        $product->product_price = $request->price;

        if($request->free_delivery)
            $product->free_delivery = 1;
        else
            $product->free_delivery = 0;

        $product->approved = 1;
        $product->save();

        return redirect()->to(env('PanelPrefix') . '/products')->with('success', $msg);
    }


    public function single()
    {

    }

    public function edit()
    {

    }

    public function delete($id): \Illuminate\Http\JsonResponse
    {
        $product = Product::query()->find($id);

        try {
            if($product->delete()){
                $result = array('status' => true , 'messege' => 'با موفقیت حذف شد');
            }
        } catch (\Illuminate\Database\QueryException $e) {
            $result = array('status' => false , 'messege' => 'حذف امکان پذیر نیست');
        }

        return response()->json($result);
    }
}
