<?php

namespace App\Http\Controllers\User;

use App\User;
use App\Product;
use App\Settings;
use App\UserFinance;
use App\ProductOrder;
use Morilog\Jalali;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class ShopController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response
     */

    public function index()
    {
        $order = ProductOrder::query()
            ->where([
                'user_id'   => Auth::id(),
                'status'    => 'cart'
            ])
            ->select('id', 'product_id', 'quantity', 'delivery_price')
            ->with(['product' => function ($query) {
                $query->select('id', 'title', 'image', 'product_price', 'quantity', 'free_delivery', 'approved');
            }])
            ->first();

        if(isset($order)) {
            if(!isset($order->product) ) {
                $order->delete();
                return redirect('product-orders');
            }
            elseif($order->product->quantity == 0 || $order->product->approved == 0)  {
                $order->delete();
                return redirect('product-orders');
            }

            if($order->product->free_delivery)
                $order->delivery_price = 0;
            else
                $order->delivery_price = Settings::query()->where('name',  'delivery_price')->value('value');

            return view('user.ecommerce.checkout', ['order' => $order]);
        }else
            return redirect('product-orders');
    }

    public function Order()
    {
        $orders = ProductOrder::query()
            ->where('user_id', Auth::id())
            ->select('id', 'product_id', 'product_title', 'final_price', 'quantity', 'status', 'payment_method', 'created_at')
            ->orderBy('updated_at', 'desc')
            ->paginate(10, ['*'], 'page_all');

        $completedOrders = ProductOrder::query()
            ->where([
                'user_id' => Auth::id(),
                'status' => 'done'
            ])
            ->select('id', 'product_id', 'final_price', 'quantity', 'status', 'payment_method', 'created_at')
            ->with(['product' => function ($query) {
                $query->select('id', 'title', 'image');
            }])
            ->orderBy('updated_at', 'desc')
            ->paginate(10, ['*'], 'page_completed');

        if(isset($orders))
            foreach ($orders as $order)
                $order->date = Jalali\CalendarUtils::strftime('H:i Y/m/d', strtotime($order->created_at));

        if(isset($completedOrders))
            foreach ($completedOrders as $order)
                $order->date = Jalali\CalendarUtils::strftime('H:i Y/m/d', strtotime($order->created_at));

        return view('user.ecommerce.shop-orders', [
            'orders' => $orders,
            'completedOrders' => $completedOrders,
        ]);
    }

    public function orderDetail($id)
    {
        $order = ProductOrder::query()
            ->where([
                'id' => $id,
                'user_id' => Auth::id(),
            ])
            ->select('id', 'user_id', 'product_id', 'product_title', 'price', 'final_price', 'delivery_price', 'admin_note', 'quantity', 'status', 'payment_method', 'created_at')
            ->with(['user' => function($query) {
                return $query->select('id', 'name', 'family', 'mobile');
            }])
            ->first();

        if(isset($order))
            $order->date = Jalali\CalendarUtils::strftime('H:i Y/m/d', strtotime($order->created_at));

        return view('user.ecommerce.order-detail', ['order' => $order]);
    }

    public function shopping(Request $request)
    {
        $orderId    = $request->get('order_id');
        $quantity   = $request->get('quantity');
        $payMethod  = $request->get('pay-method');

        $productOrder = ProductOrder::query()
            ->with(['product' => function ($query) {
                $query->select('id', 'product_price', 'quantity', 'free_delivery');
            }])
            ->find($orderId);

        $productOrder->price    = $productOrder->product->product_price;
        $productOrder->quantity = $quantity;
        $productOrder->delivery_price   = $productOrder->product->free_delivery == 0 ? Settings::query()->where('name', 'delivery_price')->value('value') : 0;
        $productOrder->final_price      = ($productOrder->price * $quantity) + $productOrder->delivery_price;
        $productOrder->payment_method   = $payMethod;

        $productOrder->save();

        $product    = Product::query()->find($productOrder->product_id);

        if($product->quantity == 0) {
            $result = array('status' => false, 'message' => 'موجودی محصول مورد نظر پایان یافته است', 'url' => '/products');
            $productOrder->status = 'canceled';
            $productOrder->save();

            return response()->json($result);
        }

        if($payMethod == 'wallet') {

            if(Auth::user()->wallet > $productOrder->final_price) {
                $user = User::query()->find(Auth::user()->id);

                $user->wallet   -= $productOrder->final_price;

                $user->save();

                $userFinance = new UserFinance();

                $userFinance->id_user   = Auth::user()->id;
                $userFinance->type      = 'برداشت';
                $userFinance->status    = 'موفق';
                $userFinance->description   = 'خرید محصول';
                $userFinance->amount    = $productOrder->final_price;
                $userFinance->stock     = $user->wallet;

                $userFinance->save();

                $productOrder->status = 'payment-confirm';

                $productOrder->save();

                //For update product quantity
                $product->quantity  = $product->quantity - $productOrder->quantity ;
                $product->save();

                $result = array('status' => true, 'message' => 'پرداخت با موفقیت انجام شد');
                return response()->json($result);
            }else {
                $result = array('status' => false, 'message' => 'موجودی شما برای خرید این محصول کافی نیست', 'url' => '/cart');
                return response()->json($result);
            }
        }
    }

    public function addToCart($productId)
    {
        $product = Product::query()->find((int) $productId);

        $exist  = ProductOrder::query()
            ->where([
                'user_id'   => auth()->id(),
                'product_id'    => $product->id
            ])
            ->whereIn('status', ['cart'])
            ->first();

        if(isset($exist)) {
            $result = array('status' => false, 'message' => 'این محصول در سبد خرید وجود دارد.');
            return response()->json($result);
        }

        $productOrder = ProductOrder::query()
            ->where('user_id', auth()->id())
            ->whereIn('status', ['cart', 'payment-pending'])
            ->first();

        if(isset($productOrder)) {
            $productOrder->status = 'canceled';
            $productOrder->save();
        }

        $productOrder = new ProductOrder();

        $productOrder->user_id      = auth()->id();
        $productOrder->product_id   = $productId;
        $productOrder->product_title   = $product->title;
        $productOrder->price        = $product->product_price;

        if($product->free_delivery)
            $productOrder->delivery_price = 0;
        else
            $productOrder->delivery_price   = Settings::query()->where('name',  'delivery_price')->value('value');

        $productOrder->final_price  =  $product->product_price + $productOrder->delivery_price;
        $productOrder->quantity     = 1;
        $productOrder->status       = 'cart';

        $productOrder->save();

        $result = array('status' => true , 'message' => 'محصول به سبد خرید شما اضافه شد');
        return response()->json($result);
    }
}
