<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateTradingTables extends Migration
{
    public function up()
    {
        // User Wallets
        Schema::create('user_wallets', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id');
            $table->string('currency', 20)->index();
            $table->decimal('balance', 20, 8)->default(0);
            $table->decimal('locked_balance', 20, 8)->default(0);
            $table->timestamps();
            
            $table->unique(['user_id', 'currency']);
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
        });
        
        // Orders
        Schema::create('orders', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id')->index();
            $table->string('market', 20)->index(); // BTCUSDT
            $table->enum('type', ['limit', 'market']);
            $table->enum('side', ['buy', 'sell']);
            $table->decimal('price', 20, 8)->nullable();
            $table->decimal('amount', 20, 8);
            $table->decimal('filled_amount', 20, 8)->default(0);
            $table->decimal('total', 20, 8)->nullable(); // price * amount
            $table->enum('status', ['pending', 'partial', 'filled', 'cancelled'])->default('pending');
            $table->string('coinex_order_id', 100)->nullable()->unique();
            $table->text('error_message')->nullable();
            $table->timestamps();
            
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
            $table->index(['user_id', 'status']);
            $table->index('created_at');
        });
        
        // Trades (Executed orders)
        Schema::create('trades', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('order_id')->nullable();
            $table->unsignedBigInteger('user_id')->index();
            $table->string('market', 20)->index();
            $table->enum('side', ['buy', 'sell']);
            $table->decimal('price', 20, 8);
            $table->decimal('amount', 20, 8);
            $table->decimal('total', 20, 8); // price * amount
            $table->decimal('fee', 20, 8)->default(0);
            $table->string('fee_currency', 20)->nullable();
            $table->string('coinex_trade_id', 100)->nullable();
            $table->timestamps();
            
            $table->foreign('order_id')->references('id')->on('orders')->onDelete('set null');
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
            $table->index('created_at');
        });
        
        // Deposits and Withdrawals
        Schema::create('deposits_withdrawals', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id')->index();
            $table->enum('type', ['deposit', 'withdrawal']);
            $table->string('currency', 20)->index();
            $table->decimal('amount', 20, 8);
            $table->decimal('fee', 20, 8)->default(0);
            $table->string('address', 255)->nullable();
            $table->string('txid', 255)->nullable()->index();
            $table->string('network', 50)->nullable(); // TRC20, ERC20, etc.
            $table->enum('status', ['pending', 'processing', 'completed', 'failed', 'cancelled'])->default('pending');
            $table->text('admin_note')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamps();
            
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
            $table->index(['user_id', 'status']);
            $table->index('created_at');
        });
        
        // KYC Verifications
        Schema::create('kyc_verifications', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id')->unique();
            $table->string('national_id', 20)->nullable();
            $table->string('first_name', 100);
            $table->string('last_name', 100);
            $table->date('birth_date')->nullable();
            $table->string('id_card_front', 255)->nullable();
            $table->string('id_card_back', 255)->nullable();
            $table->string('selfie', 255)->nullable();
            $table->enum('level', ['basic', 'intermediate', 'advanced'])->default('basic');
            $table->enum('status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->text('admin_note')->nullable();
            $table->timestamp('verified_at')->nullable();
            $table->timestamps();
            
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
        });
        
        // Trading Pairs (Markets)
        Schema::create('trading_pairs', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('market', 20)->unique(); // BTCUSDT
            $table->string('base_currency', 20); // BTC
            $table->string('quote_currency', 20); // USDT
            $table->boolean('is_active')->default(true);
            $table->decimal('min_amount', 20, 8)->default(0);
            $table->decimal('min_total', 20, 8)->default(0);
            $table->decimal('maker_fee', 10, 4)->default(0.002); // 0.2%
            $table->decimal('taker_fee', 10, 4)->default(0.002); // 0.2%
            $table->integer('price_precision')->default(8);
            $table->integer('amount_precision')->default(8);
            $table->timestamps();
            
            $table->index('is_active');
        });
    }

    public function down()
    {
        Schema::dropIfExists('trading_pairs');
        Schema::dropIfExists('kyc_verifications');
        Schema::dropIfExists('deposits_withdrawals');
        Schema::dropIfExists('trades');
        Schema::dropIfExists('orders');
        Schema::dropIfExists('user_wallets');
    }
}
