<?php
/**
 * CoinEx API Test Script
 * Test connection and basic functionality
 */

require __DIR__ . '/../vendor/autoload.php';

$app = require_once __DIR__ . '/../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = Illuminate\Http\Request::capture()
);

echo "<!DOCTYPE html>";
echo "<html><head><meta charset='UTF-8'><title>CoinEx API Test</title>";
echo "<style>body{font-family:Arial;padding:20px;max-width:1200px;margin:0 auto;}";
echo ".success{color:green;}.error{color:red;}.info{color:blue;}";
echo "table{border-collapse:collapse;width:100%;margin:10px 0;}";
echo "th,td{border:1px solid #ddd;padding:8px;text-align:left;}";
echo "th{background:#556ee6;color:white;}</style></head><body>";

echo "<h1>🔧 CoinEx API Test</h1>";

// 1. Check API Credentials
echo "<h2>1. API Credentials</h2>";
$apiKey = getSetting('coinex_api_key');
$apiSecret = getSetting('coinex_api_secret');

if ($apiKey && $apiSecret) {
    echo "<p class='success'>✅ API Key: " . substr($apiKey, 0, 10) . "..." . substr($apiKey, -5) . "</p>";
    echo "<p class='success'>✅ API Secret: " . substr($apiSecret, 0, 10) . "..." . substr($apiSecret, -5) . "</p>";
} else {
    echo "<p class='error'>❌ API credentials not configured. Please set them in Admin Panel → Settings → API Settings</p>";
    echo "</body></html>";
    exit;
}

// 2. Test CoinEx Service
echo "<h2>2. CoinEx Service</h2>";
try {
    $coinex = new \App\Services\CoinExService();
    echo "<p class='success'>✅ CoinEx Service initialized</p>";
} catch (\Exception $e) {
    echo "<p class='error'>❌ Error: " . $e->getMessage() . "</p>";
    echo "</body></html>";
    exit;
}

// 3. Test Public API - Get Markets
echo "<h2>3. Get Markets (Public API)</h2>";
try {
    $markets = $coinex->getMarkets();
    
    if (!empty($markets)) {
        echo "<p class='success'>✅ Successfully fetched " . count($markets) . " markets</p>";
        echo "<p class='info'>Showing first 10 markets:</p>";
        echo "<table><tr><th>Market</th><th>Min Amount</th><th>Pricing Name</th></tr>";
        
        $count = 0;
        foreach ($markets as $market) {
            if ($count >= 10) break;
            echo "<tr>";
            echo "<td>" . ($market['market'] ?? 'N/A') . "</td>";
            echo "<td>" . ($market['min_amount'] ?? 'N/A') . "</td>";
            echo "<td>" . ($market['pricing_name'] ?? 'N/A') . "</td>";
            echo "</tr>";
            $count++;
        }
        echo "</table>";
    } else {
        echo "<p class='error'>❌ No markets returned</p>";
    }
} catch (\Exception $e) {
    echo "<p class='error'>❌ Error: " . $e->getMessage() . "</p>";
}

// 4. Test Public API - Get Ticker
echo "<h2>4. Get Ticker for BTCUSDT (Public API)</h2>";
try {
    $ticker = $coinex->getTicker('BTCUSDT');
    
    if ($ticker) {
        echo "<p class='success'>✅ Successfully fetched ticker</p>";
        echo "<table><tr><th>Field</th><th>Value</th></tr>";
        echo "<tr><td>Last Price</td><td>" . ($ticker['last'] ?? 'N/A') . "</td></tr>";
        echo "<tr><td>24h High</td><td>" . ($ticker['high'] ?? 'N/A') . "</td></tr>";
        echo "<tr><td>24h Low</td><td>" . ($ticker['low'] ?? 'N/A') . "</td></tr>";
        echo "<tr><td>24h Volume</td><td>" . ($ticker['vol'] ?? 'N/A') . "</td></tr>";
        echo "</table>";
    } else {
        echo "<p class='error'>❌ No ticker data returned</p>";
    }
} catch (\Exception $e) {
    echo "<p class='error'>❌ Error: " . $e->getMessage() . "</p>";
}

// 5. Test Public API - Get Order Book
echo "<h2>5. Get Order Book for BTCUSDT (Public API)</h2>";
try {
    $orderBook = $coinex->getOrderBook('BTCUSDT', 5);
    
    if ($orderBook) {
        echo "<p class='success'>✅ Successfully fetched order book</p>";
        
        if (isset($orderBook['asks']) && isset($orderBook['bids'])) {
            echo "<div style='display:flex;gap:20px;'>";
            
            // Asks (Sell orders)
            echo "<div style='flex:1;'>";
            echo "<h3>Asks (Sell)</h3>";
            echo "<table><tr><th>Price</th><th>Amount</th></tr>";
            foreach (array_slice($orderBook['asks'], 0, 5) as $ask) {
                echo "<tr><td>" . $ask[0] . "</td><td>" . $ask[1] . "</td></tr>";
            }
            echo "</table></div>";
            
            // Bids (Buy orders)
            echo "<div style='flex:1;'>";
            echo "<h3>Bids (Buy)</h3>";
            echo "<table><tr><th>Price</th><th>Amount</th></tr>";
            foreach (array_slice($orderBook['bids'], 0, 5) as $bid) {
                echo "<tr><td>" . $bid[0] . "</td><td>" . $bid[1] . "</td></tr>";
            }
            echo "</table></div>";
            
            echo "</div>";
        }
    } else {
        echo "<p class='error'>❌ No order book data returned</p>";
    }
} catch (\Exception $e) {
    echo "<p class='error'>❌ Error: " . $e->getMessage() . "</p>";
}

// 6. Test Private API - Get Balances
echo "<h2>6. Get Account Balances (Private API)</h2>";
try {
    $balances = $coinex->getBalances();
    
    if (!empty($balances)) {
        echo "<p class='success'>✅ Successfully fetched balances</p>";
        echo "<table><tr><th>Currency</th><th>Available</th><th>Frozen</th></tr>";
        
        $count = 0;
        foreach ($balances as $currency => $balance) {
            if ($count >= 10) break;
            if (isset($balance['available']) && $balance['available'] > 0) {
                echo "<tr>";
                echo "<td>" . $currency . "</td>";
                echo "<td>" . $balance['available'] . "</td>";
                echo "<td>" . ($balance['frozen'] ?? 0) . "</td>";
                echo "</tr>";
                $count++;
            }
        }
        echo "</table>";
    } else {
        echo "<p class='info'>ℹ️ No balances or empty account</p>";
    }
} catch (\Exception $e) {
    echo "<p class='error'>❌ Error: " . $e->getMessage() . "</p>";
    echo "<p class='info'>This is normal if API key doesn't have trading permissions</p>";
}

echo "<hr>";
echo "<h2>📝 Summary</h2>";
echo "<p>✅ Public API endpoints are working</p>";
echo "<p>ℹ️ Private API requires proper permissions in CoinEx panel</p>";
echo "<p>🔗 <a href='https://www.coinex.com/account/api' target='_blank'>Manage API Keys in CoinEx</a></p>";

echo "</body></html>";
