<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>معاملات {{ $market }} - {{ config('app.name') }}</title>
    
    <!-- TradingView Chart Library -->
    <script src="https://unpkg.com/lightweight-charts/dist/lightweight-charts.standalone.production.js"></script>
    
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Vazir', Arial, sans-serif; background: #0d1117; color: #c9d1d9; }
        
        .trading-container { display: grid; grid-template-columns: 300px 1fr 350px; grid-template-rows: 60px 1fr; height: 100vh; gap: 1px; background: #161b22; }
        
        /* Header */
        .header { grid-column: 1 / -1; background: #161b22; padding: 0 20px; display: flex; align-items: center; justify-content: space-between; border-bottom: 1px solid #30363d; }
        .market-info { display: flex; gap: 30px; align-items: center; }
        .market-name { font-size: 20px; font-weight: bold; }
        .price { font-size: 24px; font-weight: bold; color: #58a6ff; }
        .change { padding: 4px 8px; border-radius: 4px; font-size: 14px; }
        .change.positive { background: #238636; color: #fff; }
        .change.negative { background: #da3633; color: #fff; }
        
        /* Order Book */
        .orderbook { background: #0d1117; padding: 15px; overflow-y: auto; }
        .orderbook-header { font-size: 14px; font-weight: bold; margin-bottom: 10px; display: flex; justify-content: space-between; }
        .orderbook-row { display: flex; justify-content: space-between; padding: 4px 0; font-size: 13px; cursor: pointer; }
        .orderbook-row:hover { background: #161b22; }
        .ask { color: #f85149; }
        .bid { color: #3fb950; }
        
        /* Chart */
        .chart-container { background: #0d1117; padding: 15px; }
        #chart { width: 100%; height: 100%; }
        
        /* Trading Panel */
        .trading-panel { background: #0d1117; padding: 15px; overflow-y: auto; }
        .tabs { display: flex; gap: 10px; margin-bottom: 15px; }
        .tab { padding: 8px 16px; background: #161b22; border: none; color: #c9d1d9; cursor: pointer; border-radius: 6px; }
        .tab.active { background: #238636; color: #fff; }
        
        .order-form { display: none; }
        .order-form.active { display: block; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-size: 13px; }
        .form-group input { width: 100%; padding: 10px; background: #161b22; border: 1px solid #30363d; color: #c9d1d9; border-radius: 6px; }
        
        .btn { width: 100%; padding: 12px; border: none; border-radius: 6px; font-size: 14px; font-weight: bold; cursor: pointer; margin-top: 10px; }
        .btn-buy { background: #238636; color: #fff; }
        .btn-sell { background: #da3633; color: #fff; }
        .btn:hover { opacity: 0.9; }
        
        /* Balances */
        .balances { background: #161b22; padding: 10px; border-radius: 6px; margin-bottom: 15px; }
        .balance-row { display: flex; justify-content: space-between; padding: 5px 0; font-size: 13px; }
        
        /* Orders Table */
        .orders-section { margin-top: 20px; }
        .orders-table { width: 100%; border-collapse: collapse; }
        .orders-table th { background: #161b22; padding: 8px; text-align: right; font-size: 12px; }
        .orders-table td { padding: 8px; border-top: 1px solid #30363d; font-size: 12px; }
        .cancel-btn { background: #da3633; color: #fff; border: none; padding: 4px 8px; border-radius: 4px; cursor: pointer; font-size: 11px; }
    </style>
</head>
<body>
    <div class="trading-container">
        <!-- Header -->
        <div class="header">
            <div class="market-info">
                <div class="market-name">{{ $market }}</div>
                <div class="price" id="current-price">--</div>
                <div class="change positive" id="price-change">+0.00%</div>
            </div>
            <div>
                <a href="{{ route('dashboard') }}" style="color: #58a6ff; text-decoration: none;">داشبورد</a>
            </div>
        </div>
        
        <!-- Order Book -->
        <div class="orderbook">
            <div class="orderbook-header">
                <span>قیمت</span>
                <span>مقدار</span>
            </div>
            <div id="asks-list"></div>
            <div style="text-align: center; padding: 10px; font-weight: bold;" id="spread">--</div>
            <div id="bids-list"></div>
        </div>
        
        <!-- Chart -->
        <div class="chart-container">
            <div id="chart"></div>
        </div>
        
        <!-- Trading Panel -->
        <div class="trading-panel">
            <!-- Balances -->
            <div class="balances">
                <div class="balance-row">
                    <span>{{ $tradingPair->base_currency }}:</span>
                    <span id="base-balance">{{ number_format($baseWallet->available_balance, 8) }}</span>
                </div>
                <div class="balance-row">
                    <span>USDT:</span>
                    <span id="quote-balance">{{ number_format($quoteWallet->available_balance, 2) }}</span>
                </div>
            </div>
            
            <!-- Tabs -->
            <div class="tabs">
                <button class="tab active" onclick="switchTab('limit')">Limit</button>
                <button class="tab" onclick="switchTab('market')">Market</button>
            </div>
            
            <!-- Limit Order Form -->
            <div class="order-form active" id="limit-form">
                <div class="form-group">
                    <label>قیمت (USDT)</label>
                    <input type="number" id="limit-price" step="0.01">
                </div>
                <div class="form-group">
                    <label>مقدار ({{ $tradingPair->base_currency }})</label>
                    <input type="number" id="limit-amount" step="0.00000001">
                </div>
                <div class="form-group">
                    <label>مجموع (USDT)</label>
                    <input type="number" id="limit-total" step="0.01" readonly>
                </div>
                <button class="btn btn-buy" onclick="placeOrder('limit', 'buy')">خرید</button>
                <button class="btn btn-sell" onclick="placeOrder('limit', 'sell')">فروش</button>
            </div>
            
            <!-- Market Order Form -->
            <div class="order-form" id="market-form">
                <div class="form-group">
                    <label>مقدار ({{ $tradingPair->base_currency }})</label>
                    <input type="number" id="market-amount" step="0.00000001">
                </div>
                <button class="btn btn-buy" onclick="placeOrder('market', 'buy')">خرید (Market)</button>
                <button class="btn btn-sell" onclick="placeOrder('market', 'sell')">فروش (Market)</button>
            </div>
            
            <!-- Open Orders -->
            <div class="orders-section">
                <h3 style="margin-bottom: 10px; font-size: 14px;">سفارشات باز</h3>
                <table class="orders-table" id="orders-table">
                    <thead>
                        <tr>
                            <th>نوع</th>
                            <th>قیمت</th>
                            <th>مقدار</th>
                            <th>عملیات</th>
                        </tr>
                    </thead>
                    <tbody id="orders-body">
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        const market = '{{ $market }}';
        let chart, candleSeries;
        
        // Initialize Chart
        function initChart() {
            chart = LightweightCharts.createChart(document.getElementById('chart'), {
                width: document.getElementById('chart').offsetWidth,
                height: document.getElementById('chart').offsetHeight - 30,
                layout: { background: { color: '#0d1117' }, textColor: '#c9d1d9' },
                grid: { vertLines: { color: '#161b22' }, horzLines: { color: '#161b22' } },
                timeScale: { timeVisible: true, secondsVisible: false }
            });
            
            candleSeries = chart.addCandlestickSeries({
                upColor: '#26a69a',
                downColor: '#ef5350',
                borderVisible: false,
                wickUpColor: '#26a69a',
                wickDownColor: '#ef5350'
            });
        }
        
        // Update Order Book
        function updateOrderBook(data) {
            const asksHtml = data.asks.slice(0, 15).reverse().map(ask => 
                `<div class="orderbook-row ask" onclick="setPrice(${ask[0]})">
                    <span>${parseFloat(ask[0]).toFixed(2)}</span>
                    <span>${parseFloat(ask[1]).toFixed(6)}</span>
                </div>`
            ).join('');
            
            const bidsHtml = data.bids.slice(0, 15).map(bid => 
                `<div class="orderbook-row bid" onclick="setPrice(${bid[0]})">
                    <span>${parseFloat(bid[0]).toFixed(2)}</span>
                    <span>${parseFloat(bid[1]).toFixed(6)}</span>
                </div>`
            ).join('');
            
            document.getElementById('asks-list').innerHTML = asksHtml;
            document.getElementById('bids-list').innerHTML = bidsHtml;
            
            if (data.asks[0] && data.bids[0]) {
                const spread = (parseFloat(data.asks[0][0]) - parseFloat(data.bids[0][0])).toFixed(2);
                document.getElementById('spread').textContent = `Spread: ${spread}`;
            }
        }
        
        // Update Ticker
        function updateTicker(data) {
            document.getElementById('current-price').textContent = parseFloat(data.last).toFixed(2);
            const change = parseFloat(data.change || 0);
            const changeEl = document.getElementById('price-change');
            changeEl.textContent = (change >= 0 ? '+' : '') + change.toFixed(2) + '%';
            changeEl.className = 'change ' + (change >= 0 ? 'positive' : 'negative');
        }
        
        // Fetch Real-time Data
        function fetchRealtimeData() {
            fetch(`/api/realtime/ticker/${market}`)
                .then(r => r.json())
                .then(data => { if (data.status) updateTicker(data.data); });
            
            fetch(`/api/realtime/orderbook/${market}`)
                .then(r => r.json())
                .then(data => { if (data.status) updateOrderBook(data.data); });
        }
        
        // Load Open Orders
        function loadOrders() {
            fetch(`/api/trading/my-orders?market=${market}`, {
                headers: { 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content }
            })
            .then(r => r.json())
            .then(data => {
                if (data.status) {
                    const html = data.data.map(order => `
                        <tr>
                            <td>${order.side === 'buy' ? 'خرید' : 'فروش'}</td>
                            <td>${parseFloat(order.price).toFixed(2)}</td>
                            <td>${parseFloat(order.amount).toFixed(6)}</td>
                            <td><button class="cancel-btn" onclick="cancelOrder(${order.id})">لغو</button></td>
                        </tr>
                    `).join('');
                    document.getElementById('orders-body').innerHTML = html || '<tr><td colspan="4" style="text-align:center">سفارشی وجود ندارد</td></tr>';
                }
            });
        }
        
        // Place Order
        function placeOrder(type, side) {
            const data = { market, side, type };
            
            if (type === 'limit') {
                data.price = document.getElementById('limit-price').value;
                data.amount = document.getElementById('limit-amount').value;
            } else {
                data.amount = document.getElementById('market-amount').value;
            }
            
            fetch(`/api/trading/order/${type}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify(data)
            })
            .then(r => r.json())
            .then(data => {
                alert(data.message);
                if (data.status) loadOrders();
            });
        }
        
        // Cancel Order
        function cancelOrder(orderId) {
            if (!confirm('آیا مطمئن هستید؟')) return;
            
            fetch('/api/trading/order/cancel', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({ order_id: orderId })
            })
            .then(r => r.json())
            .then(data => {
                alert(data.message);
                if (data.status) loadOrders();
            });
        }
        
        // Switch Tab
        function switchTab(tab) {
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.order-form').forEach(f => f.classList.remove('active'));
            event.target.classList.add('active');
            document.getElementById(tab + '-form').classList.add('active');
        }
        
        // Set Price from Order Book
        function setPrice(price) {
            document.getElementById('limit-price').value = price;
        }
        
        // Calculate Total
        document.getElementById('limit-price')?.addEventListener('input', () => {
            const price = parseFloat(document.getElementById('limit-price').value) || 0;
            const amount = parseFloat(document.getElementById('limit-amount').value) || 0;
            document.getElementById('limit-total').value = (price * amount).toFixed(2);
        });
        
        document.getElementById('limit-amount')?.addEventListener('input', () => {
            const price = parseFloat(document.getElementById('limit-price').value) || 0;
            const amount = parseFloat(document.getElementById('limit-amount').value) || 0;
            document.getElementById('limit-total').value = (price * amount).toFixed(2);
        });
        
        // Initialize
        initChart();
        fetchRealtimeData();
        loadOrders();
        setInterval(fetchRealtimeData, 2000);
        setInterval(loadOrders, 5000);
    </script>
</body>
</html>
